/*
 * InternalClient.java
 *
 * Created on March 7, 2005, 7:47 AM
 */

package ch.epfl.lpd.ids.server;

import ch.epfl.lpd.ids.serialization.*;
import ch.epfl.lpd.ids.utils.IBlockingQueue;
import ch.epfl.lpd.ids.utils.MyBlockingQueue;

/**
 * This class is the internal representation of a chat client for the
 * server. This class stores the username of the client and a {@link
 * ch.epfl.lpd.ids.utils.IBlockingQueue} that will be used by the
 * client to retrieve the different messages sent by the other chat
 * clients.
 * @author Sebastien Baehni
 */
public class InternalClient {
    
     /**
     * The username of this internal client.
     */
    protected String username ="";    
    
    /**
     * The {@link ch.epfl.lpd.ids.utils.IBlockingQueue} used for
     * retrieving the different messages sent by the other chat
     * clients.
     */
    protected IBlockingQueue<IMessage> blockingQueue;
    
    /**
     * Creates a new instance of InternalClient.
     * @param username The username of this internal client.
     */
    public InternalClient(String username) {
        this.username = username;        
        this.blockingQueue = new MyBlockingQueue<IMessage>();
    }

    /**
     * Method used to get the username of this internal client.
     * @return The username of this internal client.
     */
    public String getUsername() {
        return this.username;
    }
    

     /**
     * Method used to put a new {@link
     * ch.epfl.lpd.ids.serialization.IMessage} into the {@link
     * ch.epfl.lpd.ids.utils.IBlockingQueue} of this internal
     * client. This method will be called by the server when it
     * receives a new message.
     * @param msg The {@link ch.epfl.lpd.ids.serialization.IMessage}
     * to put in the queue.
     */
    public void putMessage(IMessage msg) {         
        this.blockingQueue.put(msg);
    }
    

    /**
     * Method used to retrieve the last element, not already taken,
     * that is in the {@link ch.epfl.lpd.ids.utils.IBlockingQueue} of
     * this internal client.
     * @return The last message of the {@link
     * ch.epfl.lpd.ids.utils.IBlockingQueue} of this internal client.
     */
    public IMessage getNextMessage() {
        try {            
            return this.blockingQueue.get();            
        }
        catch (InterruptedException ie) {
            ie.printStackTrace();
        }
        return new Message("An exception was caught.", "InterruptedException".getBytes());
    }
    
    
    
    
}
