/*
 * IServer.java
 *
 * Created on March 7, 2005, 7:57 AM
 */

package ch.epfl.lpd.ids.server;

import ch.epfl.lpd.ids.serialization.IMessage;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * Specification of a chat server. A chat server should be able to
 * accept new chat clients, to accept disconnection of chat clients,
 * to send/receive messages to/from these clients and to give the
 * list of its connected clients. In order to use RMI, we have to extend
 * the {@link java.rmi.Remote} class.
 * @author Sebastien Baehni
 */
public interface IServer extends Remote {
    
     /**
     * Method called by a chat client to connect to this server.
     * @param username The username of the client that wants to connect.
     * @return True if the client was able to connect, false otherwise.
     * @throws java.rmi.RemoteException If a remote exception occurs while calling this method.
     */
    public boolean connect(String username) throws RemoteException;

     /**
     * Method called by a chat client when it wants to be
     * disconnected.
     * @param username The username of the chat client that wants to
     * be disconnected.
     * @return True if the chat client was able to disconnect, false
     * otherwise.
     * @throws java.rmi.RemoteException If a remote exception occurs while calling this method.
     */
    public boolean disconnect(String username) throws RemoteException;

     /**
     * Method called by a chat client on the server when the chat
     * client wants to send a new message.
     * @param username The username of the client that sends this
     * message.
     * @param msg The {@link ch.epfl.lpd.ids.serialization.IMessage}
     * to send.
     * @throws java.rmi.RemoteException If a remote exception occurs while calling this method.
     */
    public void sendMessage(String username, IMessage msg) 
	throws RemoteException;

     /**
     * Method used to get the last message sent from a chat client to
     * this server.
     * @param username The username of the client that wants to get
     * the last message sent.
     * @return The last {@link ch.epfl.lpd.ids.serialization.IMessage}
     * sent.
     * @throws java.rmi.RemoteException If a remote exception occurs while calling this method.
     */
    public IMessage getMessage(String username) throws RemoteException;

     /**
     * Method used to retrieve the list of the connected clients.
     * @return An array of strings representing the list of the
     * connected clients.
     * @throws java.rmi.RemoteException If a remote exception occurs while calling this method.
     */
    public String[] getListOfClients() throws RemoteException;
    
    /**
     * Used when the server sends a disconnection msg to the client that wants to be disconnected.
     */
    public final String DISCONNECTION_MESSAGE = "DISCONNECTION";
}
