/*
 * ChatServer.java
 *
 * Created on March 7, 2005, 8:23 AM
 */

package ch.epfl.lpd.ids.server;

import java.util.ArrayList;

import ch.epfl.lpd.ids.serialization.IMessage;
import ch.epfl.lpd.ids.serialization.Message;
import ch.epfl.lpd.ids.utils.ServerShutdownHook;

import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.Naming;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.MalformedURLException;

/**
 * This class represents the chat server. The server manages a list of
 * connected chat clients, is able to receive messages from those
 * clients as well as to dispatch those messages to its connected
 * clients. Moreover, the clients can get a list of the connected
 * clients from this server. This class extends 
 * {@link java.rmi.server.UnicastRemoteObject} in order to be 
 * accessible remotely by the chat clients.
 * @author Sebastien Baehni
 */
public class ChatServer extends UnicastRemoteObject implements IServer {

    /**
     * A {@link java.util.ArrayList} of {@link
     * ch.epfl.lpd.ids.server.InternalClient}. 
     */
    ArrayList<InternalClient> clients;
    
     /**
     * Creates a new instance of ChatServer and initializes the {@link
     * java.util.ArrayList} of {@link
     * ch.epfl.lpd.ids.server.InternalClient}.
     * @throws java.rmi.RemoteException If there was an exception while creating the instance.
     */
    public ChatServer() throws RemoteException {
        this.clients = new ArrayList<InternalClient>();
    }

     /**
     * Method called by a client to connect to this server.
     * @param username The username of the client.
     * @return True if the connection was possible (i.e., no client
     * with the same provided username is already connected), false
     * otherwise.
     */
    public synchronized boolean connect(String username) {
        for (InternalClient client: this.clients) {
            if (client.getUsername().compareTo(username)==0) {
                return false;
            }            
        }
        InternalClient client = new InternalClient(username);
        this.clients.add(client);
        return true;
    }

    /**
     * Method called by a client to disconnect from this server.
     * @param username The username of the chat client that wants to
     * be disconnected.
     * @return True if the chat client was able to disconnect from
     * this server, false otherwise.
     */
    public synchronized boolean disconnect(String username) {
        InternalClient toRemove=null;
        for (InternalClient client: this.clients) {
            if (client.getUsername().compareTo(username)==0) {
                toRemove = client;
            }            
        }
        if (toRemove!=null) {
            IMessage msg = new Message(this.DISCONNECTION_MESSAGE, "You have successfully be disconnected from the server.".getBytes());
            toRemove.putMessage(msg);
            this.clients.remove(toRemove);
            return true;
        }
        return false;        
    }

    
    /**
     * Method called by the chat client to send the messages to this
     * server.
     * @param username The username of the client that is sending the
     * message.
     * @param msg The {@link ch.epfl.lpd.ids.serialization.IMessage}
     * sent from the client to the server.
     */
    public synchronized void sendMessage(String username, IMessage msg) {
        for (InternalClient client: this.clients) {
            client.putMessage(msg);
        }
    }
    /**
     * Method called by the chat client to get the last message sent
     * from a chat client to this server.
     * @param username The username of the client that wants to
     * retrieve the last message sent.  This username is used to wait
     * on the right {@link ch.epfl.lpd.ids.utils.IBlockingQueue} that
     * corresponds to this user.
     * @return The last {@link ch.epfl.lpd.ids.serialization.IMessage}
     * sent by a chat client to this server.
     */
    public IMessage getMessage(String username) {
        InternalClient ic = null;
        synchronized (this.clients) {
            for (InternalClient client: this.clients) {
                if (client.getUsername().compareTo(username)==0) {
                    ic = client;
                }            
            }
        }
        if (ic == null)
            return new Message("You are not connected.", "Sorry but you are not connected.".getBytes());
        else 
            return ic.getNextMessage();
    }
    
    /**
     * Method used to retrieve the list of the connected clients.
     * @return An array of strings representing the array of connected
     * clients' usernames.
     */
    public synchronized String[] getListOfClients() {
        String[] usernames = new String[this.clients.size()];
        int i=0;
        for (InternalClient client: this.clients) {
            usernames[i] = client.getUsername();
            i++;
        }
        return usernames;
    }
    
    /**
     * Utility method used to print a help usage menu.
     */
    public static void printUsage() {
        System.out.println("Usage:");
        System.out.println("java ch.epfl.lpd.ids.server.ChatServer rmiPort");        
    }
    
    /**
     * Main method used to start the server. 
     * @param args The arguments of the main. The first argument is used
     * to specify the rmi port number.
     */
    public static void main(String[] args) {   
        if (args.length < 1) {
            ChatServer.printUsage();
            System.exit(-1);
        }
        try {           
            // We add a shutdown hook to correctly stop the server.
            Runtime.getRuntime().addShutdownHook(new Thread(new ServerShutdownHook("Server successfully closed.")));
            // We create the name to which the service will be bound
            String name = "//"+InetAddress.getLocalHost().getHostName()+":"+Integer.parseInt(args[0])+"/"+"ChatServer";       
     		System.out.println(name);
            IServer is = new ChatServer();
            // We bind the service to the rmi registry.
	    Naming.rebind(name, is);            
            System.out.println("ChatServer sucessfully bound.");
        }       
        catch (UnknownHostException uhe) {
            uhe.printStackTrace();
        }
        catch (RemoteException re) {
            re.printStackTrace();
        }
        catch (MalformedURLException mue) {
            mue.printStackTrace();
        }
        
    
    }
    
}
