/*
 * ChatClient.java
 *
 * Created on March 7, 2005, 8:05 AM
 */

package ch.epfl.lpd.ids.client;

import java.util.Observable;
import java.util.Observer;

import ch.epfl.lpd.ids.gui.ClientGUI;
import ch.epfl.lpd.ids.gui.IWindowManager;
import ch.epfl.lpd.ids.server.IServer;
import ch.epfl.lpd.ids.serialization.*;

import java.rmi.RemoteException;


/**
 * This class represents a chat client. A chat client is responsible
 * for creating its GUI as well as the two threads that will fetch new
 * messages and the list of connected clients.
 * @author Sebastien Baehni
 */
public class ChatClient extends Observable implements Runnable, IChatClient {
    
    /**
     * The {@link ch.epfl.lpd.ids.gui.ClientGUI} that represents the
     * GUI of this chat client.
     */
    private ClientGUI gui;
    
    /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.server.IServer}. This reference will be used to
     * send and receive messages.
     */
    protected IServer server;
    /**
     * The reference to the {@link ch.epfl.lpd.ids.gui.IWindowManager}
     * that correctly disposes of the ressources when the client is
     * closed.
     */
    protected IWindowManager iw;
     /**
     * The username of this chat client.
     */
    protected String username = "";
    /**
     * Boolean used to check if this client is connected to the server
     * or not.
     */
    protected boolean connected = false;
    /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.client.ClientListFetcher} that is responsible
     * for retrieving the list of the connected clients.
     */
    protected ClientListFetcher clf;
    
    /**
     * Creates a new instance of ChatClient. This constructor
     * basically creates the {@link ch.epfl.lpd.ids.gui.ClientGUI},
     * the {@link ch.epfl.lpd.ids.client.ClientListFetcher} and
     * initializes the MVC.
     * @param server The {@link ch.epfl.lpd.ids.server.IServer} reference.
     * @param iw The reference to the {@link
     * ch.epfl.lpd.ids.gui.IWindowManager}.
     */
    public ChatClient(IServer server, IWindowManager iw) {        
        this.gui = new ClientGUI(this);
        this.addObserver(this.gui);
        this.server = server;
        this.iw = iw;
        this.clf = new ClientListFetcher(server,this.gui);
        this.gui.setVisible(true);        
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to connect to the server. This method calls the
     * <CODE>connect()</CODE> method on the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param username The username of the client.
     * @return True if the client was able to connect, false otherwise.
     */
    public boolean connect(String username) {
        try {
            if (!connected) {
                boolean result = this.server.connect(username);
                if (result) {
                    this.connected = true;
                    this.clf.setStarted(true);
                    new Thread(this.clf).start();
                    this.username = username;        
                    new Thread(this).start();
                }
                return result;
            }
            else {
                return false;
            }
        }
        catch (RemoteException re) {
            IMessage msg = new Message("A RemoteException Occured While Connecting.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(msg);
            re.printStackTrace();
        }
        return false;
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to disconnect from the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param username The username of the client.
     * @return True if the client was successfully disconnected, false
     * otherwise.
     */
    public boolean disconnect(String username) {
        try {           
            if (connected) {
                boolean result = this.server.disconnect(username);
                if (result) {
                    this.connected = false;
                    this.clf.setStarted(false);
                }
                return result;
            }
            else {
                return false;
            }
        }
        catch (RemoteException re) {            
            IMessage msg = new Message("A RemoteException Occured While Disconnecting.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(msg);
            re.printStackTrace();
        }
        return false;
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to send a message to the other chat
     * clients. This method creates a new {@link
     * ch.epfl.lpd.ids.serialization.IMessage} and sends it to the
     * server.
     * @param message The message as a IMessage.
     */
    public void sendMessage(IMessage msg) {
        try {
            if (connected) {
                this.server.sendMessage(this.username,msg);
            }
        }
        catch (RemoteException re) {
            IMessage tmp = new Message("A RemoteException Occured While Sending A Message.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(msg);
            re.printStackTrace();
        }
    }
    
    /**
     * Method called by {@link ch.epfl.lpd.ids.gui.ClientGUI} when the
     * user closes the chat GUI. This stops the different threads,
     * disconnects from the server and calls the {@link
     * ch.epfl.lpd.ids.gui.IWindowManager} to inform it that it can
     * release the ressources for this chat client.
     */
    public void close() {        
        this.disconnect(this.username);        
        this.iw.closeClient();
    }
           
        
    /**
     * Method called when a new thread representing this client is
     * started. This thread waits on its {@link
     * ch.epfl.lpd.ids.utils.IBlockingQueue} on the server until a new
     * message is received. Once this message is received, it is displayed
     * in the GUI.
     */
    public void run() {        
        while (connected) {
            try {
             IMessage msg = this.server.getMessage(this.username);
             this.setChanged();            
             this.notifyObservers(msg);
             if (msg.getHeader().compareTo(this.server.DISCONNECTION_MESSAGE)==0) {                
                break;
             }
            }
            catch (RemoteException re) {
                IMessage msg = new Message("A RemoteException Occured While Retrieving A Message.", re.getMessage().getBytes());
                this.setChanged();            
                this.notifyObservers(msg);
                re.printStackTrace();
            }
        }
    }     
    
     /**
     * Retrieve the username of this client.
     * @return The username of this client.
     */
    public String getUsername() {
        return this.username;
    }
    
}
