/*
 * RMIClient.java
 *
 * Created on March 7, 2005, 7:39 AM
 */

package ch.epfl.lpd.ids;

import ch.epfl.lpd.ids.server.*;
import ch.epfl.lpd.ids.serialization.*;
import ch.epfl.lpd.ids.client.*;
import ch.epfl.lpd.ids.gui.IWindowManager;
import ch.epfl.lpd.ids.utils.ClientShutdownHook;

import java.rmi.RMISecurityManager;
import java.rmi.Naming;
import java.rmi.RemoteException;
import java.rmi.NotBoundException;

import java.net.MalformedURLException;

/**
 * RMIClient class used to start the local chat.  This class starts the
 * different chat clients and connects to the remote server.
 * 
 * @author Sebastien Baehni
 */
public class RMIClient implements IWindowManager {
    
    /**
     * The different chat clients.
     */
    protected ChatClient[] clients;

    /**
     * The total number of clients to start.
     */
    protected int numberOfClients;
    
    /**
     * The port of the rmi registry.
     */
    protected int port = 1099;

     /**
     * The total number of active clients. This variable is used to
     * close the application when no more active clients are
     * available.
     */
    protected int numberOfActiveClients;
    
    /**
     * Creates a new instance of RMIClient.
     * 
     * @param numberOfClients The total number of clients to create.
     */
    public RMIClient(int numberOfClients, int port) {
        this.numberOfClients = numberOfClients;
        this.port = port;
        this.clients = new ChatClient[this.numberOfClients];
    }
    
    /**
     * Does the remote server lookup via the server host name and starts
     * the different chat clients.
     * @param serverHostName The hostname of the chat server.
     * @throws java.rmi.NotBoundException If the name of the remote object is not currently bound.
     * @throws java.net.MalformedURLException If the provided URL is malformed.
     * @throws java.rmi.RemoteException If the registry could not be contacted.
     */
    public void start(String serverHostName) throws NotBoundException,
                MalformedURLException,
                RemoteException 
    {                       
 
        // We lookup the server
        IServer server = 
	    (IServer)Naming.lookup("//"+serverHostName+":"+Integer.toString(this.port)+"/ChatServer");
        
        for (int i=0; i<this.numberOfClients; i++) {
            this.clients[i] = new ChatClient(server,this);            
            this.numberOfActiveClients++;
        }         
        Runtime.getRuntime().addShutdownHook(new Thread(new ClientShutdownHook(this.clients)));        
    }

     /**
     * Method called by a chat client when its window is closed.
     */
    public void closeClient() {        
        this.numberOfActiveClients--;
        if (this.numberOfActiveClients==0) {              
            System.exit(0);
        }
    }
    
    /**
     * Utility method used to print a help usage menu.
     */
    public static void printUsage() {
        System.out.println("Usage:");
        System.out.println("java ch.epfl.lpd.ids.RMIClient numberOfClients rmiHostName rmiPort");        
    }


     /**
     * The main method of the class. This method is the entry point of
     * the local chat application.
     * @param args The arguments given when launching the
     * application. The first argument specifies the number of clients
     * to start, the second one the server host name and the last one the 
      * rmi registry port number.
     */
    public static void main(String[] args) {
           if (args.length <3) {
               RMIClient.printUsage();
               System.exit(-1);
           }
           else {               
              // Because we download the stub of the server, we need to install
               // a security manager.
              if (System.getSecurityManager() == null) {
                System.setSecurityManager(new RMISecurityManager());
              }    
               try {
                RMIClient main = new RMIClient(Integer.parseInt(args[0]), Integer.parseInt(args[2]));
                main.start(args[1]);
               }
               catch (NumberFormatException nfe) {
                   System.out.println("Please enter a valid number of clients or a valid port number.");
                   RMIClient.printUsage();
               }
               catch (NotBoundException e) {
                   System.out.println("Unable to bound to the chat server.");
                   e.printStackTrace();
               }
               catch (MalformedURLException e) {
                   System.out.println("The hostname of the server is malformed.");
                   e.printStackTrace();
               }
               catch (RemoteException re) {
                   System.out.println("Unable to contact the rmi registry.");
                   re.printStackTrace();
               }
                
           }
    }
    
}
