/*
 * ChatServer.java
 *
 * Created on March 7, 2005, 8:23 AM
 */

package ch.epfl.lpd.ids.server;

import java.util.Vector;

import ch.epfl.lpd.ids.serialization.IMessage;
import ch.epfl.lpd.ids.serialization.Message;
import ch.epfl.lpd.ids.client.IChatClientListener;
import ch.epfl.lpd.ids.utils.ServerShutdownHook;

import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.Naming;
import java.rmi.registry.LocateRegistry;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.MalformedURLException;

import java.security.Permission;

/**
 * This class represents the chat server. The server manages a list of
 * connected chat clients, is able to receive messages from those
 * clients as well as to dispatch those messages to its connected
 * clients. Moreover, the clients can get a list of the connected
 * clients from this server. This class extends 
 * {@link java.rmi.server.UnicastRemoteObject} in order to be 
 * accessible remotely by the chat clients.
 * @author Sebastien Baehni
 */
public class ChatServer extends UnicastRemoteObject implements IServer {
    
     /**
     * A {@link java.util.Vector} of {@link java.lang.String} that 
      * represents the usernames of the clients. We use a {@link
     * java.util.Vector} instead of an {@link java.util.ArrayList}
     * because the {@link java.util.Vector} is already synchronized.
     */
    Vector<String> clientUsernames;
    
     /**
     * A {@link java.util.Vector} of {@link
     * ch.epfl.lpd.ids.client.IChatClientListener}. We use a {@link
     * java.util.Vector} instead of an {@link java.util.ArrayList}
     * because the {@link java.util.Vector} is already synchronized.
     */
    Vector<IChatClientListener> listeners;
    
    /**
     *Creates a new instance of ChatServer and initializes the {@link
     * java.util.Vector} of {@link
     * ch.epfl.lpd.ids.client.IChatClientListener} as well as the 
     * {@link java.util.Vector} containing the usernames of the clients.
     * @throws java.rmi.RemoteException If there was an exception while creating the instance.
     */
    public ChatServer() throws RemoteException {
        this.clientUsernames = new Vector<String>();
        this.listeners = new Vector<IChatClientListener>();
    }
    
    /**
     * Method called by a chat client to connect to this server.
     * @param user The username of the client that wants to connect.
     * @param iccl The {@link ch.epfl.lpd.ids.client.IChatClientListener} reference
     * in order for the server to call methods on it.
     * @return True if the client was able to connect, false otherwise.     
     */
    public boolean connect(String user, IChatClientListener iccl) {
        for (String username: this.clientUsernames) {
            if (username.compareTo(user)==0) {
                return false;
            }            
        }
        this.clientUsernames.add(user); 
        this.listeners.add(iccl);
        
        try {
            // We send the list of the connected clients to all the clients.
            for (IChatClientListener client: this.listeners) {
                client.newListOfClients(this.clientUsernames.toArray(new String[0]));
            }
        }
        catch (RemoteException re) {
            re.printStackTrace();   
            return false;
        }
        return true;
    }
    
    /**
     * Method called by a chat client when it wants to be
     * disconnected.
     * @param user The username of the chat client that wants to
     * be disconnected.
     * @param iccl The {@link ch.epfl.lpd.ids.client.IChatClientListener} reference
     * in order for the server to call methods on it.
     * @return True if the chat client was able to disconnect, false
     * otherwise.     
     */
    public boolean disconnect(String user, IChatClientListener iccl) throws RemoteException {        
        String toRemove ="";
        for (String username: this.clientUsernames) {
            if (username.compareTo(user)==0) {
                toRemove = username;
            }            
        }
        if (toRemove.compareTo("")!=0) {            
            this.clientUsernames.remove(toRemove); 
            this.listeners.remove(iccl);
            try {
                // We send a disconnection message to the disconnected client.
                IMessage msg = new Message(this.DISCONNECTION_MESSAGE, "You have successfully be disconnected from the server.".getBytes());
                iccl.newMessage(msg);
                // We update the list of the connected clients.
                for (IChatClientListener client: this.listeners) {
                    client.newListOfClients(this.clientUsernames.toArray(new String[0]));
                }                
            }
            catch (RemoteException re) {
                re.printStackTrace();
                return false;
            }
            return true;
        }
        return false;        
    }
        
    /**
     * Method called by a chat client on the server when the chat
     * client wants to send a new message.    
     * @param msg The {@link ch.epfl.lpd.ids.serialization.IMessage}
     * to send.   
     */
    public void sendMessage(IMessage msg) {
        try {
            for (IChatClientListener client: this.listeners) {
                client.newMessage(msg);
            }            
        }
        catch (RemoteException re) {
            re.printStackTrace();
        }
    }
        
    /**
     * Utility method used to print a help usage menu.
     */
     public static void printUsage() {
        System.out.println("Usage:");
        System.out.println("java ch.epfl.lpd.ids.server.ChatServer rmiPort");        
    }
    
    /**
     * Main method used to start the server. 
     * @param args The arguments of the main. The first argument is used
     * to specify the rmi port number.
     */
    public static void main(String[] args) {
        if (args.length <1) {
            ChatServer.printUsage();
            System.exit(-1);
        }
        // We set correctly the security manager.
        if (System.getSecurityManager() == null) {            
                System.setSecurityManager(new RMISecurityManager() {
                    public void checkPermission(Permission p, Object context) {                      
                    }                    
                    public void checkPermission(Permission p) {                        
                    }
            });
        }                        
        try {
            int port = Integer.parseInt(args[0]);
            try {
                // We try to create the rmi registry.
                LocateRegistry.createRegistry(port);
            }
            catch (Exception e) {
                System.out.println(e.getClass());
                // Maybe the registry is already created. In
                // this case we just get a reference to it.
                LocateRegistry.getRegistry(port);
            }
            String name = "//"+InetAddress.getLocalHost().getHostName()+":"+args[0]+"/ChatServer";
            ChatServer cs = new ChatServer();
            Naming.rebind(name, cs);
             // We add a shutdown hook to correctly stop the server.
            Runtime.getRuntime().addShutdownHook(new Thread(new ServerShutdownHook("Server successfully closed.")));
            System.out.println("ChatServer sucessfully bound.");
        }
        catch(NumberFormatException nfe) {
            nfe.printStackTrace();
            ChatServer.printUsage();
        }
        catch(UnknownHostException uhe) {
            uhe.printStackTrace();
            ChatServer.printUsage();
        }
        catch (RemoteException re) {
            re.printStackTrace();
            ChatServer.printUsage();
        }
        catch (MalformedURLException mue) {
            mue.printStackTrace();
            ChatServer.printUsage();
        }
        
    
    }
    
}
