/*
 * ChatClient.java
 *
 * Created on March 7, 2005, 8:05 AM
 */

package ch.epfl.lpd.ids.client;

import java.util.Observable;
import java.util.Observer;

import ch.epfl.lpd.ids.gui.ClientGUI;
import ch.epfl.lpd.ids.server.IServer;
import ch.epfl.lpd.ids.serialization.*;
import ch.epfl.lpd.ids.utils.ClientShutdownHook;

import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.Naming;
import java.rmi.RemoteException;
import java.rmi.NotBoundException;
import java.rmi.server.UnicastRemoteObject;

import java.net.MalformedURLException;

import java.security.Permission;


/**
 * This class represents a chat client. A chat client is responsible
 * for creating its GUI and implements the {@link ch.epfl.lpd.ids.client.IChatClientListener}
 * interface in order to be accessible remotely via RMI.
 * @author Sebastien Baehni
 */
public class ChatClient extends Observable implements IChatClientListener, IChatClient {
    
    /**
     * The {@link ch.epfl.lpd.ids.gui.ClientGUI} that represents the
     * GUI of this chat client.
     */
    private ClientGUI gui;
    
     /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.server.IServer}. This reference will be used to
     * send, connect and disconnect.
     */
    protected IServer server;
    
    /**
     * The username of this chat client.
     */
    protected String username = "";
    
    /**
     * Boolean used to check if this client is connected to the server
     * or not.
     */
    protected boolean connected = false; 
    
    /**
     * Creates a new instance of ChatClient. This constructor
     * basically creates the {@link ch.epfl.lpd.ids.gui.ClientGUI},
     * and initializes the MVC.
     * @param server The {@link ch.epfl.lpd.ids.server.IServer} reference.     
     */
    public ChatClient(IServer server) {        
        this.gui = new ClientGUI(this);
        this.addObserver(this.gui);
        this.server = server;  
        this.gui.setVisible(true);                  
    }
        
     /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to connect to the server. This method calls the
     * <CODE>connect()</CODE> method on the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param username The username of the client.
     * @return True if the client was able to connect, false otherwise.
     */
    public boolean connect(String username) {
        try {            
            if (!connected) {
                // We do not need to start threads anymore as the
                // server will call us directly.
                boolean result = this.server.connect(username,this);
                if (result) {
                    this.connected = true;                
                    this.username = username;                        
                }
                return result;
            }
            else {
                return false;
            }
        }
        catch (RemoteException re) {
            IMessage msg = new Message("A RemoteException Occured While Connecting.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(msg);
            re.printStackTrace();
        }
        return false;
    }

    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to disconnect from the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param username The username of the client.
     * @return True if the client was successfully disconnected, false
     * otherwise.
     */
    public boolean disconnect(String username) {
        try {
            if (connected) {
                boolean result = this.server.disconnect(username,this);
                if (result) {
                    this.connected = false;               
                }
                return result;
            }
            else {
                return false;
            }
        }
        catch (RemoteException re) {
            Message msg = new Message("A RemoteException Occured While Disconnecting.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(msg);
            re.printStackTrace();
        }
        return false;
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to send a message to the other chat
     * clients. This method creates a new {@link
     * ch.epfl.lpd.ids.serialization.IMessage} and sends it to the
     * server.
     * @param msg The message as a IMessage.
     */
    public void sendMessage(IMessage msg)  {
        try {
        if (connected) {
            this.server.sendMessage(msg);
        }
        }
        catch (RemoteException re) {
            IMessage tmp = new Message("A RemoteException Occured While Sending A Message.", re.getMessage().getBytes());
            this.setChanged();            
            this.notifyObservers(tmp);
            re.printStackTrace();
        }
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.server.ChatServer} to inform
     * its clients that new clients have connected.
     * @param clients The list of the clients connected to the server.
     */
    public void newListOfClients(String[] clients) {        
        this.setChanged();
        this.notifyObservers(clients);        
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.server.ChatServer} to inform
     * its clients that a new message has been sent.
     * @param msg The message that has been sent by one client to the server.
     */
    public void newMessage(IMessage msg) throws RemoteException {
        this.setChanged();
        this.notifyObservers(msg);        
    }
    
    /**
     * Method called by {@link ch.epfl.lpd.ids.gui.ClientGUI} when the
     * user closes the chat GUI. It disconnects from the server and 
     * stops the application.
     */
    public void close() {
        this.disconnect(this.username);        
        System.exit(0);
    }     
    
    /**
     * Retrieve the username of this client.
     * @return The username of this client.
     */
    public String getUsername() {
        return this.username;
    }
          
    /**
     * Method used to print a help usage.
     * Please note that the codebase is not given in the
     * help message.
     */
    public static void printUsage() {
        System.out.println("Usage:");
        System.out.println("java ch.epfl.lpd.ids.client.ChatClient rmiHostName rmiPort");        
    }
    
    /**
     * Main method used to launch the {@link ch.epfl.lpd.ids.client.ChatClient} 
     * application. We set a new security manager and we export the 
     * {@link ch.epfl.lpd.ids.client.IChatClientListener} interface in order
     * to load the stub of the client.
     * @param args The first argument represents the hostname of the {@link ch.epfl.lpd.ids.server.ChatServer}
     * and the second argument is the port of the RMI registry to connect.
     */
    public static void main(String[] args) {
           if (args.length <2) {
               ChatClient.printUsage();
               System.exit(1);
           }
           else {
               
               if (System.getSecurityManager() == null) {                   
                    System.setSecurityManager(new RMISecurityManager() {
                    public void checkPermission(Permission p, Object context) {
                    }                    
                    public void checkPermission(Permission p) {
                    }
                });
               }               
               try {
                   int port = Integer.parseInt(args[1]);
                   // We lookup the ChatServer service
                   IServer server = (IServer)Naming.lookup("//"+args[0]+":"+Integer.toString(port)+"/ChatServer");
                   IChatClientListener iccl = new ChatClient(server);   
                   // We export our ChatClient object in order to load the stub
                   // of the client such that the server can connect and use it.
                   UnicastRemoteObject.exportObject(iccl);      
                    // A hook to correctly disconnect the client from the server.
                    Runtime.getRuntime().addShutdownHook(new Thread(new ClientShutdownHook((IChatClient)iccl)));   
               }
               catch (NumberFormatException nfe) {
                   nfe.printStackTrace();
                   ChatClient.printUsage();
               }
               catch (NotBoundException e) {
                e.printStackTrace();
                ChatClient.printUsage();
               }
               catch (MalformedURLException e) {
                   e.printStackTrace();
                   ChatClient.printUsage();
               }
               catch (RemoteException re) {
                   re.printStackTrace();
                   ChatClient.printUsage();
               }                               
           }
    }
    
}
