package ch.epfl.lpd.ids.client;

import java.util.Observable;
import java.util.Observer;

import ch.epfl.lpd.ids.server.IServer;

/**
 * Thread used to retrieve the list of the connected clients. This
 * thread is started/stopped whenever the user clicks on the
 * connect/disconnect button of the {@link
 * ch.epfl.lpd.ids.gui.ClientGUI}.
 * @author Sebastien Baehni
 */
public class ClientListFetcher extends Observable implements Runnable {
    
    /**
     * Boolean to know if the thread has been started or not.
     */
    protected boolean started = false;
    /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.server.IServer}. This reference is used to
     * retrieve the list of connected clients.
     */
    protected IServer server;
    
    /**
     * Creates a new instance of ClientListFetcher.
     * @param server The reference to the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param observer The {@link ch.epfl.lpd.ids.gui.ClientGUI} that
     * is update evertime a list of clients is retrieved.
     */
    public ClientListFetcher(IServer server, 
                            Observer observer) {
        this.server = server;
        this.addObserver(observer);
    }
    
    /**
     * Setter for the started boolean
     * @param bool The boolean used to set the thread to be started or
     * not.
     */
    public void setStarted(boolean bool) {
        this.started = bool;
    }
    
    /**
     * Main run method that fetch the list of connected chat clients
     * on the server.  This method does not block and polls the server
     * every three seconds. Usually it is better to do passive
     * waiting, but for this special case, we can use active waiting.
     */
    public void run() {
        while (started) {
            try {
                Thread.currentThread().sleep(3000);
                if (started){
                    String[] listOfClients = this.server.getListOfClients();
                    this.setChanged();            
                    this.notifyObservers(listOfClients);     
                }
            }
            catch (InterruptedException ie) {
                ie.printStackTrace();
            }
            
        }
        
    }
    
}
