package ch.epfl.lpd.ids.client;

import java.util.Observable;
import java.util.Observer;

import ch.epfl.lpd.ids.gui.ClientGUI;
import ch.epfl.lpd.ids.gui.IWindowManager;
import ch.epfl.lpd.ids.server.IServer;
import ch.epfl.lpd.ids.serialization.*;

/**
 * This class represents a chat client. A chat client is responsible
 * for creating its GUI as well as the two threads that will fetch new
 * messages and the list of connected clients.
 * @author Sebastien Baehni
 */
public class ChatClient extends Observable implements Runnable, IChatClient {
    
    /**
     * The {@link ch.epfl.lpd.ids.gui.ClientGUI} that represents the
     * GUI of this chat client.
     */
    private ClientGUI gui;
    /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.server.IServer}. This reference will be used to
     * send and receive messages.
     */
    protected IServer server;
    /**
     * The reference to the {@link ch.epfl.lpd.ids.gui.IWindowManager}
     * in order to correctly dispose the ressources when the client is
     * closed.
     */
    protected IWindowManager iw;
    /**
     * The username of this chat client.
     */
    protected String username = "";
    /**
     * Boolean used to check is this client is connected to the server
     * or not.
     */
    protected boolean connected = false;
    /**
     * The reference to the {@link
     * ch.epfl.lpd.ids.client.ClientListFetcher} that is responsible
     * for retrieving the list of the connected clients.
     */
    protected ClientListFetcher clf;
    
    /**
     * Creates a new instance of ChatClient. This constructor
     * basically creates the {@link ch.epfl.lpd.ids.gui.ClientGUI},
     * the {@link ch.epfl.lpd.ids.client.ClientListFetcher} and
     * initialize the MVC.
     * @param server The {@link ch.epfl.lpd.ids.server.IServer} reference.
     * @param iw The reference to the {@link
     * ch.epfl.lpd.ids.gui.IWindowManager}.
     */
    public ChatClient(IServer server, IWindowManager iw) {        
        this.gui = new ClientGUI(this);
        this.addObserver(this.gui);
        this.server = server;
        this.iw = iw;
        this.clf = new ClientListFetcher(server,this.gui);
        this.gui.setVisible(true);        
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to connect to the server. This method calls the
     * <CODE>connect()</CODE> method on the {@link
     * ch.epfl.lpd.ids.server.IServer}
     * @param username The username of the client.
     * @return True if the client was able to connect, false otherwise.
     */
    public boolean connect(String username) {
        if (!this.connected) {
            this.connected = this.server.connect(username);
            if (this.connected) {
                this.clf.setStarted(true);
                new Thread(this.clf).start();
                this.username = username;        
                new Thread(this).start();
            }
        }
        return this.connected;
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to disconnect from the {@link
     * ch.epfl.lpd.ids.server.IServer}.
     * @param username The username of the client.
     * @return True if the client was successfully disconnected, false
     * otherwise.
     */
    public boolean disconnect(String username) {
        if (connected) {
            this.connected = false;
            boolean result = this.server.disconnect(username);
            if (result) {                
                this.clf.setStarted(false);
            }
            else {
                this.connected = true;
            }
            return result;
        }
        else {
            return false;
        }
    }
    
    /**
     * Method called by the {@link ch.epfl.lpd.ids.gui.ClientGUI} when
     * the user wants to send a message to the other chat
     * clients. This method creates a new {@link
     * ch.epfl.lpd.ids.serialization.IMessage} and sends it to the
     * server.
     * @param msg The message as a IMessage.
     */
    public void sendMessage(IMessage msg) {
        if (connected) {
            this.server.sendMessage(this.username,msg);
        }
    }
    
    /**
     * Method called by {@link ch.epfl.lpd.ids.gui.ClientGUI} when the
     * user close the chat GUI. This stops the different threads,
     * disconnects from the server and calls the {@link
     * ch.epfl.lpd.ids.gui.IWindowManager} to inform it that it can
     * release the ressources for this chat client.
     */
    public void close() {
        this.disconnect(this.username);
        this.clf.setStarted(false);
        this.iw.closeClient();
    }
           
        
    
    /**
     * Method called when a new thread representing this client is
     * started. This thread wait on its {@link
     * ch.epfl.lpd.ids.utils.IBlockingQueue} on the server until a new
     * message is received. Once this message is received, it is put
     * in the GUI.
     */
    public void run() {        
        while (connected) {
            IMessage msg = this.server.getMessage(this.username);
            this.setChanged();            
            this.notifyObservers(msg); 
        }      
    }
    
}
