package ch.epfl.lpd.ids;

import ch.epfl.lpd.ids.server.*;
import ch.epfl.lpd.ids.serialization.*;
import ch.epfl.lpd.ids.client.*;
import ch.epfl.lpd.ids.gui.IWindowManager;

/**
 * CentralizedChat class used to start the local chat. 
 * This class starts the server and the different chat clients.
 * 
 * @author Sebastien Baehni
 */
public class CentralizedChat implements IWindowManager {
    
    /**
     * The different chat clients.
     */
    protected ChatClient[] clients;
    /**
     * The total number of clients to start.
     */
    protected int numberOfClients;
    /**
     * The total number of active clients. This variable is used to
     * close the application when no more active clients are
     * available.
     */
    protected int numberOfActiveClients;
    
    /**
     * Creates a new instance of CentralizedChat.
     * 
     * @param numberOfClients The total number of clients to create
     */
    public CentralizedChat(int numberOfClients) {
        this.numberOfClients = numberOfClients;
        this.clients = new ChatClient[this.numberOfClients];
    }
    
    /**
     * Creates the chat server object and the different chat clients.
     */
    public void start() {                       
        IServer server = new ChatServer();
        for (int i=0; i<this.numberOfClients; i++) {
            this.clients[i] = new ChatClient(server,this);            
            this.numberOfActiveClients++;
        }                         
    }
    
    /**
     * Method called by a chat client when its window is closed.
     */
    public void closeClient() {
        this.numberOfActiveClients--;
        if (this.numberOfActiveClients==0) {
            System.exit(0);
        }
    }
    
    /**
     * Utility method used to print a help usage menu.
     */
    public static void printUsage() {
        System.out.println("Usage:");
        System.out.println("java ch.epfl.lpd.ids.Main numberOfClients");
    }
    
    /**
     * The main method of the class. This method is the entry point of
     * the local chat application.
     * @param args The arguments given when launching the
     * application. The first and only argument specifies the number
     * of clients to start.
     */
    public static void main(String[] args) {
           if (args.length <1) {
               CentralizedChat.printUsage();
               System.exit(1);
           }
           else {
               try {
                CentralizedChat main = new CentralizedChat(Integer.parseInt(args[0]));
                main.start();
               }
               catch (NumberFormatException nfe) {
                   System.out.println("Please enter a valid number.");
                   CentralizedChat.printUsage();
               }
                
           }
    }
    
}
